<#/*****************************PARAMETER SECTION STARTS*****************************/
param (
    [String] $driverName = "PostgreSQL Unicode(x64)",
    [String] $server,
    [String] $dataBase,
    [String] $port,
    [String] $username,
    [String] $password,
    [String] $tableName,
    [String] $columnNames,
    [String] $columnValues
)
/*****************************PARAMETER SECTION ENDS*****************************/#>

<#/*****************************FUNCTION SECTION STARTS*****************************/#>
# Function to create ODBC Connection with Database
Function createOdbcConnection([String]$dsn) {
    try {
        $conn = New-Object System.Data.Odbc.OdbcConnection
        $conn.ConnectionString = "DSN=$dsn;"
        openConnection $conn
        return $conn;
    } catch {
        Write-Host "Error in Creating ODBC Connection for the specified datasource named " + $dsn
        Write-Host $_.Exception.Message 
        $false
    }
}

# Function to check whether database connection is open
Function isConnectionOpen([System.Data.Odbc.OdbcConnection]$conn) {
    try {
        if($conn.State -eq "Open") {
            $true
        } else {
            $false
        }
    } catch {
        Write-Host "Error in Checking connection state to the specified database"
        Write-Host $_.Exception.Message 
        $false
    }
}

# Function to close the connection with database
Function closeConnection([System.Data.Odbc.OdbcConnection]$conn) {
    try {
        $conn.Close()
    } catch {
        Write-Host "Error in Closing connection to the specified database"
        Write-Host $_.Exception.Message 
        $false
    }
}

# Function to open the connection with database
Function openConnection([System.Data.Odbc.OdbcConnection]$conn) {
    try {
        $conn.Open()
    } catch {
        Write-Host "Error in Opening connection to the specified database"
        Write-Host $_.Exception.Message 
        $false
    }
}

# Function to execute the DB query into database
Function executeNonQuery([System.Data.Odbc.OdbcConnection]$conn, [string]$query) {
    try {
        $cmd = New-Object System.Data.Odbc.OdbcCommand($query,$conn)
        $cmd.ExecuteNonQuery()
    } catch {
        Write-Host "Error in Executing the Query"
        Write-Host $_.Exception.Message 
        $false
    }    
}

# Function to construct Update query
Function getUpdateQuery([String] $tableName, [String[]] $columnNames, [String[]] $columnValues, [String] $whereQuery) {
    try {
        $query = "UPDATE $tableName SET"
        For ($i=0; $i -lt $columnNames.Length; $i++) {
            $query += " " + $columnNames[$i] + " = " 
            if($columnValues[$i] -eq '') {
                $query += "null"
            } elseif($columnValues[$i] -like '^s.*') {
                $query += "'"+$columnValues[$i].substring(3)+"'";
            } else {
                $query += $columnValues[$i]
            }
            if($i -lt ($columnNames.Length - 1)){
			    $query += ", ";
            }
        }
        $query += " WHERE $whereQuery;"
        return $query
    } catch {
        Write-Host "Error in Creating the Updation Query"
        Write-Host $_.Exception.Message 
        return ""
    }
}

# Function to get Seperated values based on a seperator
Function getSeperatedValues([String] $string, [String] $separator) {
    return $string.Split($separator);
}
<#/*****************************FUNCTION SECTION ENDS*****************************/#>

<#/*****************************MAIN SCRIPT SECTION STARTS*****************************/#>
try {

    # Getting the args
    $driverName = $args[0]
    $server = $args[1]
    $dataBase = $args[2]
    $port = $args[3]
    $username = $args[4]
    $password = $args[5]
    $tableName = $args[6]
    $columnNames = $args[7]
    $columnValues = $args[8]
    $whereQuery = $args[9]

	$dsn = 'PostgreSQL';
    $dsnType = 'System';
	$trustedConnection = 'Yes';
    # Setting ODBC Data Source Property Values
    $propertyValues = New-Object System.Collections.ArrayList
    $temp = $propertyValues.Add("Server=" + $server); 
    $temp = $propertyValues.Add("Trusted_Connection=" + $trustedConnection); 
    $temp = $propertyValues.Add("Database=" + $dataBase); 
    $temp = $propertyValues.Add("port=" + $port); 
    $temp = $propertyValues.Add("username=" + $username); 
    $temp = $propertyValues.Add("password=" + $password); 

    # Adding ODBC Data Source
    Add-OdbcDsn -Name $dsn -DriverName $driverName -DsnType $dsnType -SetPropertyValue $propertyValues

    # Creating ODBC Connection
    $conn = createOdbcConnection $dsn
    if(isConnectionOpen $conn ) {

        # Processing Column Names and Column Values
        $columnNamesArr = getSeperatedValues $columnNames "|"
        $columnValuesArr = getSeperatedValues $columnValues "|"

        #Constructing Query
        $query = getUpdateQuery $tableName $columnNamesArr $columnValuesArr $whereQuery

        if($query -ne "") {
            # Executing Query
            executeNonQuery $conn $query
        }
        
    }
} catch {
    Write-Host $_.Exception.Message 
    $false
} finally {
    closeConnection $conn

    # Removing ODBC Data Source
    Remove-OdbcDsn -Name $dsn -DsnType $dsnType
}
<#/*****************************MAIN SCRIPT SECTION STARTS*****************************/#>